/**
* \file: watch.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <sys/epoll.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>

#include "control/watch.h"
#include "utils/logger.h"

#define POLL_TIMEOUT_INFINITE -1

static int watch_pollfd=-1;

static void watch_process_event(struct epoll_event *event);

//---------------------------------------- API members ---------------------------------------------
error_code_t watch_init(void)
{
	error_code_t result=RESULT_OK;

	if (watch_pollfd!=-1)
		watch_deinit();

	watch_pollfd=epoll_create1(EPOLL_CLOEXEC);
	if (watch_pollfd < 0)
	{
		logger_log_error("WATCH - Unable to create an epoll fd.");
		result=RESULT_NORESOURCE;
	}

	return result;
}

void watch_deinit(void)
{
	if (watch_pollfd!=-1)
	{
		close(watch_pollfd);
		watch_pollfd=-1;
	}
}

error_code_t watch_add_event_source(watch_t *watch, uint32_t event_mask)
{
	error_code_t result=RESULT_OK;
	struct epoll_event ev;

	ev.events = event_mask;
	ev.data.ptr=watch;
	watch->event_mask=event_mask;

	if (epoll_ctl(watch_pollfd, EPOLL_CTL_ADD, watch->pollfd, &ev) < 0)
	{
		logger_log_error("WATCH - Failed to add the poll file descriptor to the watch poll fd.");
		result=RESULT_NORESOURCE;
	}

	return result;
}

error_code_t watch_modify_watch_events(watch_t *watch, uint32_t event_mask)
{
	struct epoll_event ev;

	ev.events = event_mask;
	ev.data.ptr=watch;

	if (event_mask==watch->event_mask)
		return RESULT_OK;

	if (epoll_ctl(watch_pollfd, EPOLL_CTL_MOD, watch->pollfd, &ev) < 0)
	{
		if (errno==ENOENT)
			logger_log_error("WATCH - The watch is not registered.");
		else
			logger_log_error("WATCH - Failed to modify the event mask of the watch.");

		return RESULT_INVALID;
	}
	watch->event_mask=event_mask;

	return RESULT_OK;
}

error_code_t watch_remove_event_source(watch_t *watch)
{
	if (epoll_ctl(watch_pollfd, EPOLL_CTL_DEL, watch->pollfd, NULL) < 0)
	{
		logger_log_error("WATCH - Failed to remove the poll file descriptor from the watch poll fd.");
		return RESULT_NORESOURCE;
	}

	return RESULT_OK;
}

watch_process_event_result_t watch_wait_for_and_process_event(void)
{
	int epoll_result;
	int poll_sleep_to = POLL_TIMEOUT_INFINITE;

	struct epoll_event event;

	if (watch_pollfd==-1)
		return WATCH_ERROR;

	logger_log_debug("WATCH - wait for events now ...");

	epoll_result = epoll_wait(watch_pollfd, &event, 1, poll_sleep_to);
	if (epoll_result < 0)
	{
		if (errno != EINTR)
		{
			logger_log_error("WATCH - Problems with the pollfd.");
			return WATCH_ERROR;
		}
		else
			return WATCH_INTERRUPTED;
	}
	else
	{
		if (epoll_result>0)
		{
			watch_process_event(&event);
			return WATCH_EVENT_PROCESSED;
		}
		else
			return WATCH_TIMED_OUT;
	}
}
//-------------------------------------------------------------------------------------------------

//------------------------------------- private members -------------------------------------------
static void watch_process_event(struct epoll_event *event)
{
	watch_t *watch;
	watch=(watch_t *)event->data.ptr;
	logger_log_debug("WATCH - Now processing event.");
	if (watch->callback_func!=NULL)
		watch->callback_func(watch, event->events);
}
//-------------------------------------------------------------------------------------------------
